/*
 * Copyright (C) 2020 Logical Clocks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud;

import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.hadoop.hdfs.protocol.CloudBlock;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

public interface CloudPersistenceProvider {
  /*
  deletes all the bucket belonging to the user.
  This is only used for testing.
   */
  public void deleteAllBuckets(String prefix) throws IOException, ExecutionException, InterruptedException;


  public boolean existsCID(String bucket) throws IOException;

  public void setCID(String bucket, String cid) throws IOException;

  public String getCID(String bucket) throws IOException;

  public boolean isEmpty(String bucket) throws IOException;

  public boolean bucketExists(String bucket) throws IOException;

  /*
   Deletes all hopsfs releated data
   */
  public void format(List<String> buckets) throws IOException;

  /*
  Create bucket
   */
  public void createBucket(String bucket) throws IOException;
  /*
  Check that all the buckets needed exist
  throws runtime exception if the buckets dont exists or not writable.
   */
  public void checkAllBuckets(List<String> buckets) throws IOException;

  public int getPrefixSize();

  public void uploadObject(String bucket, String objectKey, File object,
                           Map<String, String> metadata) throws IOException;

  public boolean objectExists(String bucket, String objectKey)
          throws IOException;

  public Map<String, String> getUserMetaData(String bucket, String objectKey)
          throws IOException;

  public long getObjectSize(String bucket, String objectKey)
          throws IOException;

  public void downloadObject(String bucket, String objectKey, File path)
          throws IOException;

  public Map<BlockIDAndGSTuple, CloudBlock> getAll(String prefix, List<String> buckets) throws IOException;

  public List<String> getAllHopsFSDirectories(List<String> buckets) throws IOException;

  public void deleteObject(String bucket, String objectKey) throws IOException;

  public void renameObject(String srcBucket, String dstBucket, String srcKey,
                           String dstKey) throws IOException ;

 public void copyObject(String srcBucket, String dstBucket, String srcKey,
                           String dstKey, Map<String, String> newObjMetadata) throws IOException ;

  public long getPartSize();

  public int getXferThreads();

  public UploadID startMultipartUpload(String bucket, String objectKey,
                                     Map<String, String> metadata)
          throws IOException;

  public PartRef uploadPart(String bucket, String objectKey, UploadID uploadID,
                             int partNo, File file, long startPos, long endPos)
          throws IOException;

  public void finalizeMultipartUpload(String bucket, String objectKey,
                                      UploadID uploadID, List<PartRef> partETags)
          throws IOException;

  public void abortMultipartUpload(String bucket, String objectKey, UploadID uploadID)
          throws IOException;

  public List<ActiveMultipartUploads> listMultipartUploads(List<String> buckets, String prefix) throws IOException;

  boolean restoreDeletedBlock(String bucket, String objectKey) throws IOException;

  boolean isVersioningSupported(String bucket) throws IOException;

  public void deleteAllVersions(String bucket, String objectKey) throws IOException;

  public void deleteOldVersions(String bucket, String objectKey) throws IOException;

  public void shutdown();

  public Object getCloudClient();
}
