/*
 * Copyright (C) 2020 LogicalClocks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.datanode;

import com.amazonaws.services.s3.model.PartETag;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.PartRef;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.UploadID;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Future;

public class ProvidedReplicaBeingWritten extends ReplicaBeingWritten {

  // this is set when we start using multiparts
  private boolean isMultipart = false;

  // to check if all parts have been uploaded
  private boolean isMultipartComplete = false;

  // this indicates that the multipart was cancelled by
  // the client by calling hsync/hflush
  private boolean cancellMultipart = false;

  // this is set if the block is opened for append
  private boolean isAppend = false;

  // old versions of the block
  private  List<Long>  oldGS = new ArrayList<>();

  // this is set if the block is opened for recovery
  private boolean isRecovered = false;

  private int currentPart = 0;
  private final long partSize;
  List<PartRef> partETags = new ArrayList<PartRef>();
  List<Future> uploadTasks = new ArrayList<Future>();
  private UploadID uploadID = null;

  // if the client calls sync operation before the blocks is closed then
  // the block is uploaded to the cloud and this param is set
  private boolean isSynced;

  public ProvidedReplicaBeingWritten(ReplicaBeingWritten from, long partSize) {
    super(from);
    this.partSize = partSize;
    this.isSynced = false;
  }

  public boolean isPartAvailable() {
    // if the client calls hsync/hflush then
    // the block is written to the cloud immediately
    // and we can not use multipart upload to upload
    // parts of the block in parallel

    if (cancellMultipart) {
      return false;
    }

    if (getBytesOnDisk() >= ((currentPart + 1) * partSize)) {
      return true;
    }
    return false;
  }

  public int incrementAndGetNextPart() {
    return ++currentPart;
  }

  public int getCurrentPart() {
    return currentPart;
  }

  public long getPartSize() {
    return partSize;
  }

  public UploadID getUploadID() {
    return uploadID;
  }

  public void setUploadID(UploadID uploadID) {
    this.uploadID = uploadID;
  }

  public void addEtag(PartRef etag){
    partETags.add(etag);
  }

  public List<PartRef> getPartETags(){
    return partETags;
  }

  public boolean isMultipart() {
    return isMultipart;
  }

  public void setMultipart(boolean multipart) {
    isMultipart = multipart;
  }

  public boolean isMultipartComplete() {
    return isMultipartComplete;
  }

  public void setMultipartComplete(boolean multipartComplete) {
    isMultipartComplete = multipartComplete;
  }

  public List<Future> getAllUploadTasks(){
    return uploadTasks;
  }

  public void addUploadTask(Future future){
    uploadTasks.add(future);
  }

  public boolean isSynced() {
    return isSynced;
  }

  public void setSynced(boolean synced) {
    isSynced = synced;
  }

  public boolean isCancellMultipart() {
    return cancellMultipart;
  }

  public void setCancellMultipart(boolean cancellMultipart) {
    this.cancellMultipart = cancellMultipart;
  }

  public boolean isAppend() {
    return isAppend;
  }

  public void setAppend(boolean append) {
    isAppend = append;
  }

  public void setRecovered(boolean recovered) {
    this.isRecovered = recovered;
  }

  public boolean isRecovered() {
    return isRecovered;
  }

  public List<Long> getOldGS() {
    return oldGS;
  }

  public void addOldGS(long oldGS) {
    this.oldGS.add(oldGS);
  }
}
