/*
 * Copyright (C) 2024 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.blockmanagement;

import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.metadata.hdfs.entity.INodeIdentifier;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.StringUtils;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.UUID;
import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


public class ProvidedBlocksCorruptionChecker implements Callable<Object> {

  public static final Log LOG = LogFactory.getLog(ProvidedBlocksCorruptionChecker.class);
  private static int HEADER_LEN = 7;
  final Namesystem ns;
  final BlockInfoContiguous block;
  final DatanodeStorageInfo storageInfo;
  final String reason;
  final CorruptReplicasMap.Reason reasonCode;
  final Configuration conf;

  public ProvidedBlocksCorruptionChecker(Namesystem ns,
                                         Configuration conf,
                                         BlockInfoContiguous block,
                                         DatanodeStorageInfo storageInfo,
                                         String reason,
                                         CorruptReplicasMap.Reason reasonCode) {
    this.ns = ns;
    this.block = block;
    this.conf = conf;
    this.storageInfo = storageInfo;
    this.reason = reason;
    this.reasonCode = reasonCode;
  }

  @Override
  public Object call() throws Exception {

    CloudPersistenceProvider cloudConnector = null;
    try {
      cloudConnector =
              CloudPersistenceProviderFactory.getCloudClient(conf);

      if (hasCheckSumError(cloudConnector)) {
        markCorrupt();
      } else {
        deleteFromDNCache(); // the block may be bad in the DN cache
      }

    } catch (Exception e) {
      LOG.info("HopsFS-Cloud: Block checksum verification failed " + e);
    } finally {
      if (cloudConnector != null) {
        cloudConnector.shutdown();
      }
    }
    return null;
  }

  /*
  Return true if there is checksum error.
  All other errors raise exception
   */
  boolean hasCheckSumError(CloudPersistenceProvider cloudConnector) throws IOException {
    String bucket = block.getCloudBucket();
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    String blockKey = CloudHelper.getBlockKey(prefixSize, block);
    String metaKey = CloudHelper.getMetaFileKey(prefixSize, block);

    File blockFile = null,   metaFile = null;
    FileInputStream metaStream = null, dataStream = null;
    FileChannel metaChannel = null, dataChannel = null;
    DataInputStream checksumStream = null;
    try {
      File tmp = new File(System.getProperty("java.io.tmpdir"));
      blockFile = new File(tmp, UUID.randomUUID().toString());
      metaFile = new File(tmp, UUID.randomUUID().toString());

      //download the block and meta file from cloud
      cloudConnector.downloadObject(bucket, blockKey, blockFile);
      cloudConnector.downloadObject(bucket, metaKey, metaFile);

      BlockMetadataHeader header;
      try {
        metaStream = new FileInputStream(metaFile);
        checksumStream = new DataInputStream(metaStream);
        header = BlockMetadataHeader.readHeader(checksumStream);
        metaChannel = metaStream.getChannel();
        metaChannel.position(HEADER_LEN);
      } catch (RuntimeException e) {
        throw new IOException("Failed to read HDFS metadata file header for " +
                metaFile + ": " + StringUtils.stringifyException(e));
      } catch (IOException e) {
        throw new IOException("Failed to read HDFS metadata file header for " +
                metaFile + ": " + StringUtils.stringifyException(e));
      }
      DataChecksum checksum = header.getChecksum();
      LOG.debug("Checksum type: " + checksum.toString());
      ByteBuffer metaBuf, dataBuf;
      try {
        dataStream = new FileInputStream(blockFile);
        dataChannel = dataStream.getChannel();
        final int CHECKSUMS_PER_BUF = 1024 * 32;
        metaBuf = ByteBuffer.allocate(checksum.
                getChecksumSize() * CHECKSUMS_PER_BUF);
        dataBuf = ByteBuffer.allocate(checksum.
                getBytesPerChecksum() * CHECKSUMS_PER_BUF);
      } catch (IOException e) {
        throw new IOException("Failed to open HDFS block file for " +
                blockFile + ": " + StringUtils.stringifyException(e));
      }
      long offset = 0;
      while (true) {
        dataBuf.clear();
        int dataRead = -1;
        try {
          dataRead = dataChannel.read(dataBuf);
          if (dataRead < 0) {
            break;
          }
        } catch (IOException e) {
          throw new IOException("Got I/O error reading block file " +
                  blockFile + "from disk at offset " + dataChannel.position() +
                  ": " + StringUtils.stringifyException(e));
        }
        try {
          int csumToRead =
                  (((checksum.getBytesPerChecksum() - 1) + dataRead) /
                          checksum.getBytesPerChecksum()) *
                          checksum.getChecksumSize();
          metaBuf.clear();
          metaBuf.limit(csumToRead);
          metaChannel.read(metaBuf);
          dataBuf.flip();
          metaBuf.flip();
        } catch (IOException e) {
          throw new IOException("Got I/O error reading metadata file " +
                  metaFile + "from disk at offset " + metaChannel.position() +
                  ": " + StringUtils.stringifyException(e));
        }
        try {
          checksum.verifyChunkedSums(dataBuf, metaBuf,
                  blockFile.getAbsolutePath(), offset);
        } catch (IOException e) {
          LOG.info("verifyChunkedSums error: " +
                  StringUtils.stringifyException(e));
          return true;
        }
        offset += dataRead;
      }
      LOG.info("Checksum verification succeeded on block " + block);
      return false;
    } finally {
      IOUtils.cleanup(null, metaStream, dataStream, checksumStream);
      if (blockFile != null && blockFile.exists()) {
        blockFile.delete();
      }
      if (metaFile != null && metaFile.exists()) {
        metaFile.delete();
      }
    }
  }

  private void markCorrupt() throws IOException {
    new HopsTransactionalRequestHandler(HDFSOperationType.CLOUD_ADD_CORRUPT_BLOCKS) {
      INodeIdentifier inodeIdentifier;

      @Override
      public void setUp() throws StorageException {
        inodeIdentifier = INodeUtil.resolveINodeFromBlock(block);
      }

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        LockFactory lf = LockFactory.getInstance();
        locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE, inodeIdentifier))
                .add(lf.getIndividualBlockLock(block.getBlockId(), inodeIdentifier))
                .add(lf.getBlockRelated(LockFactory.BLK.UR));
      }

      @Override
      public Object performTask() throws IOException {
        ((FSNamesystem) ns).getBlockManager().
                getNeededReplications().add(block, 0, 0, (int) 1);
        return null;
      }
    }.handle();
  }

  void deleteFromDNCache() throws IOException {

    new HopsTransactionalRequestHandler(HDFSOperationType.FIND_AND_MARK_BLOCKS_AS_CORRUPT) {
      INodeIdentifier inodeIdentifier;

      @Override
      public void setUp() throws StorageException {
        inodeIdentifier = INodeUtil.resolveINodeFromBlock(block);
      }

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        LockFactory lf = LockFactory.getInstance();
        locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE, inodeIdentifier))
                .add(lf.getIndividualBlockLock(block.getBlockId(), inodeIdentifier))
                .add(lf.getBlockRelated(LockFactory.BLK.IV));
      }

      @Override
      public Object performTask() throws IOException {
        final BlockInfoContiguous storedBlock = ((FSNamesystem) ns).getStoredBlock(block);
        if (storedBlock == null) {
          LOG.info("HopsFS-Cloud: BLOCK* findAndMarkBlockAsCorrupt: " + block + " not found");
          // cleaned by block report
          return null;
        }

        ((FSNamesystem) ns).getBlockManager().getInvalidateBlocks().addProvidedBlock(block,
                inodeIdentifier.getInodeId(), false/*removeFromCloud*/);
        return null;
      }
    }.handle();
  }
}
