/*
 * Copyright (C) 2019, 2024 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.lc.repackaged.com.google.cloud.storage.Blob;
import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.leaderElection.LeaderElection;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.HdfsVariables;
import io.hops.metadata.Variables;
import io.hops.metadata.common.entity.LongVariable;
import io.hops.metadata.common.entity.Variable;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.BlockLookUpDataAccess;
import io.hops.metadata.hdfs.dal.ProvidedBlockReportTasksDataAccess;
import io.hops.metadata.hdfs.dal.UnderReplicatedBlockDataAccess;
import io.hops.metadata.hdfs.entity.*;
import io.hops.transaction.EntityManager;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.ActiveMultipartUploads;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.GCSActiveMultipartUploads;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.S3ActiveMultipartUploads;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import static org.apache.hadoop.hdfs.DFSConfigKeys.*;

/*
 Compare the blocks in the DB and in the cloud bucket(s)
 */
public class ProvidedBlocksChecker extends Thread {
  private final Namesystem ns;
  private boolean run = true;
  private final int prefixSize;
  private final long blockReportDelay;
  private final long sleepInterval;
  private final long subTaskSize;
  private final long markBlocksCorruptOrMissingDelay;
  private final BlockManager bm;
  private final int maxProvidedBRThreads;
  private boolean isBRInProgress = false;
  private final long deleteAbandonedBlocksAfter;
  private final Configuration conf;
  private final boolean isS3MultipartUpload;
  private final boolean isGCSMultipartUpload;

  CloudPersistenceProvider cloudConnector;

  static final Log LOG = LogFactory.getLog(ProvidedBlocksChecker.class);

  public ProvidedBlocksChecker(Configuration conf, Namesystem ns, BlockManager bm) throws IOException {
    this.ns = ns;
    this.bm = bm;
    this.conf = conf;
    this.prefixSize = conf.getInt(
            DFS_CLOUD_PREFIX_SIZE_KEY,
            DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    this.blockReportDelay = conf.getLong(
            DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
            DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
    this.sleepInterval = conf.getLong(
            DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY,
            DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_DEFAULT);
    this.subTaskSize = conf.getLong(DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY,
            DFS_CLOUD_BR_SUB_TASKS_SIZE_DEFAULT);
    this.markBlocksCorruptOrMissingDelay = conf.getLong(
            DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY,
            DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_DEFAULT);
    this.maxProvidedBRThreads = conf.getInt(DFS_CLOUD_MAX_BR_THREADS_KEY,
            DFS_CLOUD_MAX_BR_THREADS_DEFAULT);
    this.deleteAbandonedBlocksAfter =
            conf.getLong(DFS_CLOUD_DELETE_ABANDONED_MULTIPART_FILES_AFTER,
            DFS_CLOUD_DELETE_ABANDONED_MULTIPART_FILES_AFTER_DEFAUlT);

    if (subTaskSize % prefixSize != 0) {
      String message = "Invalid configuration " + DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY
              + " must be multiple of " + DFS_CLOUD_PREFIX_SIZE_KEY;
      LOG.error(message);
      throw new IllegalArgumentException(message);
    }

    String cloudProvider = conf.get(
            DFSConfigKeys.DFS_CLOUD_PROVIDER,
            DFSConfigKeys.DFS_CLOUD_PROVIDER_DEFAULT);
    //only s3 supports checking for failed multipart uploads
    //Azure automatically deletes the failed uploads
    this.isS3MultipartUpload = conf.getBoolean(
            DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD,
            DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD_DEFAUlT) &&
            cloudProvider.compareToIgnoreCase(CloudProvider.AWS.name()) == 0;
    this.isGCSMultipartUpload = conf.getBoolean(
            DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD,
            DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD_DEFAUlT) &&
            cloudProvider.compareToIgnoreCase(CloudProvider.GCS.name()) == 0;

    this.cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);

  }

  /*
   * If Leader
   * ---------
   *    If it is time to block report then add all the sub tasks in the queue
   *
   * All nodes
   *    Poll the BR tasks queue and find work if any
   */
  @Override
  public void run() {
    while (run) {
      try {

        if (ns.isRunning() && ns.isLeader()) {
          long startTime = getProvidedBlocksScanStartTime();
          long existingTasks = countBRPendingTasks();
          long timeElapsed = System.currentTimeMillis() - startTime;

          if (timeElapsed > blockReportDelay && existingTasks == 0) {
            final long END_ID = HdfsVariables.getMaxBlockID();
            List<ProvidedBlockReportTask> tasks = generateTasks(END_ID);
            LOG.info("HopsFS-Cloud. BR Created " + tasks.size() + " block " +
                "reporting tasks for block ids up to " + END_ID);
            addNewBlockReportTasks(tasks);
            // every block reporting cycle check for aborted S3
            // multipart uploads. Azure automatically clears the failed
            // multipart uploads. For GCS this is implemented at
            // prefix (dir) level
            checkS3AbandonedBlocks();
          }
        }

        // poll for work
        if (countBRPendingTasks() != 0) {
          startWork();
        }

      } catch (IOException e) {
        LOG.warn(e, e);
      } finally {
        if (run) {
          try {
            Thread.sleep(sleepInterval);
          } catch (InterruptedException e) {
            currentThread().interrupt();
          }
        }
      }
    }
  }

  private void startWork() throws IOException {
    //start workers and wait for them to finish their work
    Collection workers = new ArrayList<>();
    for (int i = 0; i < maxProvidedBRThreads; i++) {
      workers.add(new BRTasksPullers(i));
    }

    try {
      isBRInProgress = true;
      List<Future<Object>> futures =
              ((FSNamesystem) ns).getBlockOperationsExecutor().invokeAll(workers);
      //Check for exceptions
      for (Future<Object> maybeException : futures) {
        maybeException.get();
      }

    } catch (InterruptedException e) {
      LOG.error(e.getMessage(), e);
      throw new IOException(e);
    } catch (ExecutionException e) {
      if (e.getCause() instanceof IOException) {
        throw (IOException) e.getCause();
      } else {
        throw new IOException(e.getCause());
      }
    } finally {
      isBRInProgress = false;
    }
  }

  private boolean processTask(ProvidedBlockReportTask task) throws IOException {
    boolean successful = false;
    try {
      for (long start = task.getStartIndex(); start < task.getEndIndex(); ) {
        long end = start + prefixSize;
        String prefix = CloudHelper.getPrefix(prefixSize, start);
        LOG.debug("HopsFS-Cloud. BR Checking prefix: " + prefix);
        Map<BlockIDAndGSTuple, BlockInfoContiguous> dbBlocksMap = new HashMap<>();
        Map<Long, UnderReplicatedBlock> corruptBlkMap = new HashMap<>();
        findAllBlocksRange(start, end, dbBlocksMap, corruptBlkMap);
        ProvidedBlocksCheckerFaultInjector.get().errorAfterReadingBlocks(dbBlocksMap);
        Map<BlockIDAndGSTuple, CloudBlock> cloudBlocksMap = cloudConnector.getAll(prefix,
                Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
        LOG.debug("HopsFS-Cloud. BR DB view size: " + dbBlocksMap.size() +
                " Cloud view size: " + cloudBlocksMap.size());

        List<BlockInfoContiguous> toMissing = new ArrayList<>();
        List<BlockToMarkCorrupt> toCorrupt = new ArrayList<>();
        List<CloudBlock> toDelete = new ArrayList<>();
        List<BlockInfoContiguous> toUnCorrupt = new ArrayList<>();
        reportDiff(dbBlocksMap, cloudBlocksMap, corruptBlkMap, toMissing, toCorrupt, toDelete, toUnCorrupt);
        if ( toMissing.size() != 0 || toCorrupt.size() != 0 || toDelete.size() != 0 ) {
          LOG.info("HopsFS-Cloud. BR toMissing: " + toMissing.size() +
                  " toCorrupt: " + toCorrupt.size() +
                  " toDelete: " + toDelete.size() + " Prefix: " + prefix);
        }
        handleMissingBlocks(toMissing);
        handleCorruptBlocks(toCorrupt);
        handleToDeleteBlocks(toDelete);
        handleToUnCorruptBlocks(toUnCorrupt);
        //"prefix" already ends with /
        handleGCSFailedMultipartUploads(prefix+CloudHelper.GCS_MULTI_PART_DIR);

        start = end;
      }
      successful = true;
    } catch (Exception e) {
      LOG.warn(e, e);
    } finally {
      return successful;
    }
  }

  public List<ProvidedBlockReportTask> generateTasks(long maxBlkID) {
    List<ProvidedBlockReportTask> tasks = new ArrayList<>();
    long startIndex = 0;
    while (startIndex < maxBlkID) {
      long endIndex = (startIndex) + subTaskSize;
      ProvidedBlockReportTask task = new ProvidedBlockReportTask(
              startIndex,
              endIndex,
              0, LeaderElection.LEADER_INITIALIZATION_ID);
      tasks.add(task);
      startIndex = endIndex;
    }
    return tasks;
  }

  /**
   * This class is used to pull work from the queue and execute the operation
   */
  class BRTasksPullers implements Callable {
    private int id;
    private  int count;
    BRTasksPullers(int id){
      this.id  = id;
    }
    @Override
    public Object call() throws Exception {
      ProvidedBlockReportTask task = null;
      long startTime = System.currentTimeMillis();
      do {
        task = popPendingBRTask();
        if (task != null) {
          processTask(task);
          count++;
        } else {
          LOG.info("HopsFS-Cloud. BR Worker ID: "+id+" processed "+count+" tasks. Total " +
                  "Processing time: "+(System.currentTimeMillis()- startTime)+" ms.");
          return null;
        }
      } while (!currentThread().isInterrupted());
      return null;
    }
  }


  /*
   * A block is marked corrupt when
   *    1. If the block's metadata exists in the database, but the block is absent form the
   *       bucket listing.
   *    2. Partial block information was obtained using cloud block listing. In S3 ls operation
   *       is eventually consistent. That is, it is possible that for a block the ls operation
   *       might return the meta block information but miss the actual block, or vice versa.
   *       Such blocks are marked corrupt for the time being
   *    3. The generations stamp, block size or bucket id does not match
   *
   *  A block is marked for deletion if it is in the cloud bucket listing but no corresponding
   *    metadata is present in the database.
   */
  @VisibleForTesting
  public void reportDiff(Map<BlockIDAndGSTuple, BlockInfoContiguous> dbView,
                         Map<BlockIDAndGSTuple, CloudBlock> cView,
                         Map<Long, UnderReplicatedBlock> corruptView,
                         List<BlockInfoContiguous> toMissing, List<BlockToMarkCorrupt> toCorrupt,
                         List<CloudBlock> toDelete, List<BlockInfoContiguous> toUnCorrupt) throws IOException {
    final Set<BlockIDAndGSTuple> aggregatedSafeBlocks = new HashSet<>();
    aggregatedSafeBlocks.addAll(dbView.keySet());

    for (BlockIDAndGSTuple dbBlockKey : dbView.keySet()) {
      BlockInfoContiguous dbBlock = dbView.get(dbBlockKey);
      CloudBlock cblock = cView.get(dbBlockKey);
      BlockToMarkCorrupt cb = null;

      //under construction blocks should be handled using normal block reports.
      if(dbBlock instanceof  BlockInfoContiguousUnderConstruction){

        if(cView.containsKey(dbBlockKey)) {
          // this can happen if the client calls hsync/hflush
          // or if the RECEIVED_BLOCK message has not been processed by the NN
          cView.remove(dbBlockKey);
        }

        aggregatedSafeBlocks.remove(dbBlock.getBlockId());
        continue;
      }

      if (cblock == null  || cblock.isPartiallyListed()) {
        // if the block is partially listed  or totally missing form the listing
        // then only mark the block missing aka corrupt after Δt, because s3 is eventually
        // consistent

        //take most recent time stamp
        long lastModifiedTime = dbBlock.getTimestamp();
        if (cblock != null && cblock.getLastModified() > lastModifiedTime) {
          lastModifiedTime = cblock.getLastModified();
        }

        if ((System.currentTimeMillis() - lastModifiedTime) > markBlocksCorruptOrMissingDelay) {
          toMissing.add(dbBlock);
          aggregatedSafeBlocks.remove(dbBlock.getBlockId()); //this block is not safe now
        }
        cView.remove(dbBlockKey);

        LOG.info("HopsFS-Cloud: BR delayed action on DB: " + dbBlock + " Cloud: " + ((cblock == null) ? "null" : cblock));
        continue;
      }
      // No need to check matching of GSs, as GS is part of the key in the hashMap.
      // We will not get here if GS does not match
      // Checking for block size and bucket names

      else if (cblock.getBlock().getNumBytes() != dbBlock.getNumBytes()) {
        cb = new BlockToMarkCorrupt(cblock, dbBlock, "Block size mismatch",
                CorruptReplicasMap.Reason.SIZE_MISMATCH);
      } else if (cblock.getBlock().getCloudBucket().compareToIgnoreCase(dbBlock.getCloudBucket()) != 0) {
        cb = new BlockToMarkCorrupt(cblock, dbBlock, "Cloud bucket mismatch",
                CorruptReplicasMap.Reason.INVALID_STATE);
      } else {
        //not corrupt
        cView.remove(dbBlockKey);

        // check if it was previously marked as corrupt.
        // For example, the block store returns empty list
        // due to internal error.
        // Encountered this problem during S3 outage.
        // Mark the blocks as not corrupt as everything matches
        if (corruptView.containsKey(dbBlockKey.getBlockID())) {
          toUnCorrupt.add(dbBlock);
        }
      }

      if (cb != null) {
        toCorrupt.add(cb);
        aggregatedSafeBlocks.remove(dbBlock.getBlockId()); //this block is not safe now
        cView.remove(dbBlockKey);
      }
    }

    // assume the GS of the block is 1002 in the database
    // while it is 1003 in the cloud.
    // This can happen if the client has updated the GS on the DN side and the
    // corresponding (updatePipeline and BLOCK_RECEIVED) messages
    // have not yet been processed by the NN. The client has
    // either died or the block report has run before the updatePipeline/
    // updateBlockForPipeline messages are processed by the NN
    //
    // As the keys in the hash maps above include generation stamp(see above)
    // the database block  will be marked missing
    // as there is no corresponding block in the cloud with the same GS
    // Similarly, the cloud block be marked for deletion as it does not belong
    // to any block in the database.
    for (BlockIDAndGSTuple tuple : cView.keySet()) {
      CloudBlock cloudBlock = cView.get(tuple);

      //if this belongs to a block that is under construction then it is safe
      //to ignore this until the block is closed by the client or the lease manager
      if(belongsToBlkUC(dbView, cloudBlock)){
        continue;
      }

      BlockInfoContiguous dbBlock = removeFromDBMissing(cloudBlock.getBlock().getBlockId(), toMissing);
      boolean dbMissingFromCloud =  dbBlock == null;
      boolean timeout =
              (System.currentTimeMillis() - cloudBlock.getLastModified()) > markBlocksCorruptOrMissingDelay;
      if(dbMissingFromCloud && timeout){
        // we can safely delete this block
        toDelete.add(cloudBlock);
      } else if (!dbMissingFromCloud && timeout) {
        // GS mismatch. Mark it as corrupt. Do not delete the block from the cloud
        toCorrupt.add( new BlockToMarkCorrupt(cloudBlock, dbBlock, "Block GS mismatch",
                CorruptReplicasMap.Reason.GENSTAMP_MISMATCH));
      } else  {
        // Delay taking any action until timeout. Maybe some corresponding
        // messages are in flight and not processed and not processed yet.
        // When it times out it will be marked for deletion or marked as corrupt
        LOG.info("HopsFS-Cloud: BR delayed action on DB: "+dbBlock+" Cloud: "+cloudBlock);
      }
    }

    //safe blocks
    if (ns.isInStartupSafeMode()) {
      LOG.info("HopsFS-Cloud. BR Aggregated safe block #: " + aggregatedSafeBlocks.size());
      final Set<Long> aggregatedSafeBlocksIDs = new HashSet<>();
      for(BlockIDAndGSTuple tuple : aggregatedSafeBlocks){
        aggregatedSafeBlocksIDs.add(tuple.getBlockID());
      }
      ns.adjustSafeModeBlocks(aggregatedSafeBlocksIDs);
    }
  }

  private boolean belongsToBlkUC(Map<BlockIDAndGSTuple, BlockInfoContiguous> dbView,
                                 CloudBlock cb) {
    for (BlockInfoContiguous bc : dbView.values()) {
      if (bc.getBlockId() == cb.getBlock().getBlockId()) {
        if (bc.getBlockUCState() == HdfsServerConstants.BlockUCState.COMPLETE) {
          return false;
        } else {
          return  true;
        }
      }
    }
    return false;
  }

  private BlockInfoContiguous removeFromDBMissing(Long blockID,
                                    List<BlockInfoContiguous> missingBlocks) {
    int found = -1;
    for(int i = 0; i < missingBlocks.size(); i++){
      BlockInfoContiguous block = missingBlocks.get(i);
      if(block.getBlockId() == blockID){
        found = i;
      }
    }
    if( found != -1 ){
      return missingBlocks.remove(found);
    } else {
      return null;
    }
  }

  //In this case missing blocks are also considered corrupt.
  //We will never recover from this state hence corruption
  private void handleMissingBlocks(final List<BlockInfoContiguous> missingBlocks)
          throws IOException {
    for (BlockInfoContiguous blk : missingBlocks) {
      LOG.info("HopsFS-Cloud: BR Marking block " + blk + " corrupt because it is missing");
      addCorrptUnderReplicatedBlock(blk);
    }
  }

  private void handleCorruptBlocks(List<BlockToMarkCorrupt> corruptBlocks)
          throws IOException {
    for (BlockToMarkCorrupt b : corruptBlocks) {
      LOG.info("HopsFS-Cloud: BR Marking block " + b.stored + " corrupt because " + b.reason);
      addCorrptUnderReplicatedBlock(b.stored);
    }
  }

  private void handleToUnCorruptBlocks(List<BlockInfoContiguous> unCorruptBlocks)
          throws IOException {
    for (BlockInfoContiguous b : unCorruptBlocks) {
      LOG.info("HopsFS-Cloud: BR Marking block " + b + " as not corrupt");
      removeCorruptUnderReplicatedBlock(b);
    }
  }

  private void addCorrptUnderReplicatedBlock(final BlockInfoContiguous block) throws IOException {
    new HopsTransactionalRequestHandler(HDFSOperationType.CLOUD_ADD_CORRUPT_BLOCKS) {
      INodeIdentifier inodeIdentifier;

      @Override
      public void setUp() throws StorageException {
        inodeIdentifier = INodeUtil.resolveINodeFromBlock(block);
      }

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        LockFactory lf = LockFactory.getInstance();
        locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE, inodeIdentifier));
        locks.add(lf.getIndividualBlockLock(block.getBlockId(), inodeIdentifier))
                .add(lf.getBlockRelated(LockFactory.BLK.UR));
      }

      @Override
      public Object performTask() throws StorageException, IOException {
        bm.neededReplications.add(block, 0, 0, 1);
        return null;
      }
    }.handle();
  }

  private void removeCorruptUnderReplicatedBlock(final BlockInfoContiguous block) throws IOException {
    new HopsTransactionalRequestHandler(HDFSOperationType.CLOUD_ADD_CORRUPT_BLOCKS) {
      INodeIdentifier inodeIdentifier;

      @Override
      public void setUp() throws StorageException {
        inodeIdentifier = INodeUtil.resolveINodeFromBlock(block);
      }

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        LockFactory lf = LockFactory.getInstance();
        locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE, inodeIdentifier));
        locks.add(lf.getIndividualBlockLock(block.getBlockId(), inodeIdentifier))
                .add(lf.getBlockRelated(LockFactory.BLK.UR));
      }

      @Override
      public Object performTask() throws StorageException, IOException {
        bm.neededReplications.remove(block);
        return null;
      }
    }.handle();
  }

  @VisibleForTesting
  public int handleToDeleteBlocks(List<CloudBlock> toDelete) throws IOException {
    List<InvalidatedBlock> invblks = new ArrayList<>();
    int count = 0;
    for (CloudBlock cblock : toDelete) {
      if(handleToDeleteBlock(cblock)){
        count ++;
      }
    }
    return count;
  }

  private boolean handleToDeleteBlock(final CloudBlock cblock) throws IOException {

    HopsTransactionalRequestHandler processDeletedBlock =
            new HopsTransactionalRequestHandler(
                    HDFSOperationType.DELETE_CLOUD_BLKS_BY_BLK_RPT) {
              INodeIdentifier inodeIdentifier;

              @Override
              public void setUp() throws StorageException {
                inodeIdentifier = INodeUtil.resolveINodeFromBlock(cblock.getBlock());
              }

              @Override
              public void acquireLock(TransactionLocks locks) throws IOException {
                LockFactory lf = LockFactory.getInstance();
                if (inodeIdentifier != null) {
                  locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE,
                          inodeIdentifier, true));
                  locks.add(lf.getIndividualBlockLock(cblock.getBlock().getBlockId(),
                          inodeIdentifier));
                  locks.add(lf.getBlockRelated(LockFactory.BLK.IV));
                }
              }

              @Override
              public Object performTask() throws IOException {
                boolean delete = true;
                if (inodeIdentifier != null) {
                  INode file = EntityManager.find(INode.Finder.ByINodeIdFTIS,
                          inodeIdentifier.getInodeId());
                  if (file == null) {
                    LOG.info("HopsFS-Cloud. BR. Deleting block that does not belong to any file. " +
                            "Block: " + cblock);
                  } else {
                    BlockInfoContiguous blockInfo =
                            EntityManager.find(BlockInfoContiguous.Finder.ByBlockIdAndINodeId,
                                    cblock.getBlock().getBlockId(), inodeIdentifier.getInodeId());

                    if (blockInfo != null && blockInfo.getGenerationStamp() == cblock.getBlock().getGenerationStamp()) {
                      LOG.warn("HopsFS-Cloud. BR. Ignoring delete request for" +
                              " block ID: " + blockInfo.getBlockId() + " GS: " + blockInfo.getGenerationStamp() +
                              " inode ID: " + file.getId());
                      delete = false;
                    }
                  }
                }

                if (delete) {
                  LOG.info("HopsFS-Cloud. BR Deleting block id=" + cblock.getBlock());
                  Block block = cblock.getBlock();
                  InvalidatedBlock invBlk = new InvalidatedBlock(
                          StorageId.CLOUD_STORAGE_ID,
                          block.getBlockId(),
                          block.getGenerationStamp(),
                          CloudHelper.getCloudBucketID(block.getCloudBucket()),
                          Long.MAX_VALUE, // do not send back a response
                          INode.NON_EXISTING_INODE_ID, true);
                  List<InvalidatedBlock> invblks = new ArrayList<>();
                  invblks.add(invBlk);
                  bm.getInvalidateBlocks().addAll(invblks);
                  return true;
                }
                return false;
              }
            };
    return (boolean)processDeletedBlock.handle();
  }


  public void shutDown() {
    run = false;
    interrupt();
  }

  public void findAllBlocksRange(final long startID, final long endID,
                            Map<BlockIDAndGSTuple, BlockInfoContiguous> blkMap,
                            Map<Long, UnderReplicatedBlock> corruptBlkMap)
          throws IOException {
    HopsTransactionalRequestHandler handler =
            new HopsTransactionalRequestHandler(HDFSOperationType.BR_GET_RANGE_OF_BLOCKS) {

              @Override
              public void acquireLock(TransactionLocks locks) throws IOException {
              }

              @Override
              public Object performTask() throws IOException {

                //get the primary keys of the blocks in the range
                BlockLookUpDataAccess da = (BlockLookUpDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockLookUpDataAccess.class);
                long allBlockIDs[] = new long[(int) (endID - startID)];

                int index = 0;
                for (long blockID = startID; blockID < endID; blockID++, index++) {
                  allBlockIDs[index] = blockID;
                }

                long allInodeIDs[] = da.findINodeIdsByBlockIds(allBlockIDs);
                assert allInodeIDs.length == allBlockIDs.length;

                List<Long> blockIdsArr = new LinkedList<>();
                List<Long> inodeIdsArr = new LinkedList<>();

                for(int i = 0; i < allInodeIDs.length; i++){
                  if(allInodeIDs[i] > INodeDirectory.ROOT_INODE_ID){
                    inodeIdsArr.add(allInodeIDs[i]);
                    blockIdsArr.add(allBlockIDs[i]);
                  }
                }

                // Get cloud blocks
                getCloudBlock(blockIdsArr.stream().mapToLong(l -> l).toArray(),
                        inodeIdsArr.stream().mapToLong(l -> l).toArray(), blkMap );

                // Get corrupt cloud blocks
                getCorruptCloudBlocks(blkMap, corruptBlkMap);

                return null;
              }
            };
    handler.handle();
  }

  private void getCloudBlock(long[] blockIDs,
                             long[] inodeIDs,
                             Map<BlockIDAndGSTuple, BlockInfoContiguous> outBlkMap) throws StorageException {
    BlockInfoDataAccess bda = (BlockInfoDataAccess) HdfsStorageFactory
            .getDataAccess(BlockInfoDataAccess.class);
    List<BlockInfoContiguous> existingBlocks = bda.findByIds(blockIDs, inodeIDs);

    for (BlockInfoContiguous blk : existingBlocks) {
      if (blk.isProvidedBlock()) {
        //This class can only handle blocks stored in the cloud
        outBlkMap.put(new BlockIDAndGSTuple(blk.getBlockId(), blk.getGenerationStamp()), blk);
      }
    }
  }

  private void getCorruptCloudBlocks(Map<BlockIDAndGSTuple, BlockInfoContiguous> inBlkMap,
                                Map<Long, UnderReplicatedBlock> outCorruptMap) throws StorageException {
    // list of cloud blocks marked corrupt
    long inodeIDs[] = new long[inBlkMap.size()];
    long blkIDs[] = new long[inBlkMap.size()];
    int index = 0;
    for (BlockInfoContiguous blk : inBlkMap.values()) {
      inodeIDs[index] = blk.getInodeId();
      blkIDs[index] = blk.getBlockId();
      index++;
    }

    UnderReplicatedBlockDataAccess urbda =
            (UnderReplicatedBlockDataAccess) HdfsStorageFactory.getDataAccess(UnderReplicatedBlockDataAccess.class);
    List<UnderReplicatedBlock> corruptBlocks = urbda.findByIds(inodeIDs, blkIDs);

    for (UnderReplicatedBlock blk : corruptBlocks) {
      outCorruptMap.put(blk.getBlockId(), blk);
    }
  }

  public class BlockToMarkCorrupt {
    /**
     * The corrupted block in a datanode.
     */
    final CloudBlock corrupted;
    /**
     * The corresponding block stored in the BlockManager.
     */
    final BlockInfoContiguous stored;
    /**
     * The reason to mark corrupt.
     */
    final String reason;
    /**
     * The reason code to be stored
     */
    final CorruptReplicasMap.Reason reasonCode;

    BlockToMarkCorrupt(CloudBlock corrupted,
                       BlockInfoContiguous stored, String reason,
                       CorruptReplicasMap.Reason reasonCode) {
      Preconditions.checkNotNull(stored, "stored is null");

      this.corrupted = corrupted;
      this.stored = stored;
      this.reason = reason;
      this.reasonCode = reasonCode;
    }

    @Override
    public String toString() {
      return stored + " Reason: " + reason;
    }
  }

  public static void scheduleBlockReportNow() throws IOException {
    LOG.debug("HopsFS-Cloud. BR Scheduling a block report now");
    setProvidedBlocksScanStartTime(0L);
  }

  public ProvidedBlockReportTask popPendingBRTask()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_POP_TASK) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        ProvidedBlockReportTasksDataAccess da = (ProvidedBlockReportTasksDataAccess) HdfsStorageFactory
                .getDataAccess(ProvidedBlockReportTasksDataAccess.class);
        ProvidedBlockReportTask task = (ProvidedBlockReportTask) da.popTask();
        LOG.debug("HopsFS-Cloud. BR pulled a task from queue Task: " + task);
        return task;
      }
    };
    return (ProvidedBlockReportTask) handler.handle();
  }

  public long countBRPendingTasks()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_COUNT_TASKS) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        ProvidedBlockReportTasksDataAccess da = (ProvidedBlockReportTasksDataAccess) HdfsStorageFactory
                .getDataAccess(ProvidedBlockReportTasksDataAccess.class);
        return da.count();
      }
    };
    return (long) handler.handle();
  }

  public List<ProvidedBlockReportTask> getAllTasks()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_GET_ALL_TASKS) {

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        Variables.getVariable(Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        ProvidedBlockReportTasksDataAccess da = (ProvidedBlockReportTasksDataAccess) HdfsStorageFactory
                .getDataAccess(ProvidedBlockReportTasksDataAccess.class);
        return da.getAllTasks();
      }
    };
    return (List<ProvidedBlockReportTask>) handler.handle();
  }


  public void addNewBlockReportTasks(
          final List<ProvidedBlockReportTask> tasks)
          throws IOException {

    // if the number of tasks are more more than 1000 then add the task
    // in small batches. Add large number of rows may fail the transaction
    for(int index = 0; index < tasks.size();) {
      int start = index;
      int end = (index + 1000) > tasks.size() ? tasks.size() : (index + 1000);
      addNewBlockReportTasks(tasks.subList(start, end), start == 0);
      index = end;
    }
  }

  public void addNewBlockReportTasks(
          final List<ProvidedBlockReportTask> tasks, final boolean updateCounters)
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_ADD_TASKS) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        ProvidedBlockReportTasksDataAccess da = (ProvidedBlockReportTasksDataAccess) HdfsStorageFactory
                .getDataAccess(ProvidedBlockReportTasksDataAccess.class);
        da.addTasks(tasks);

        if(LOG.isDebugEnabled()) {
          for (ProvidedBlockReportTask task : tasks) {
            LOG.debug("HopsFS-Cloud. BR Added new block report tasks " + task);
          }
        }

        if(updateCounters) {
          long startTime = System.currentTimeMillis();
          Variables.updateVariable(
                  new LongVariable(
                          Variable.Finder.providedBlocksCheckStartTime,
                          startTime
                  ));
          if (LOG.isDebugEnabled()) {
            LOG.debug("HopsFS-Cloud. BR set start time to : " +
                    new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(startTime)));
          }

          long counter = (long) Variables.getVariable(
                  Variable.Finder.providedBlockReportsCount).getValue();
          Variables.updateVariable(
                  new LongVariable(
                          Variable.Finder.providedBlockReportsCount,
                          counter + 1
                  ));
          LOG.debug("HopsFS-Cloud. BR set counter to : " + (counter + 1));
        }

        return null;
      }
    };
    handler.handle();
  }

  public long getProvidedBlockReportsCount()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_COUNT_TASKS) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        long counter = (long) Variables.getVariable(
                Variable.Finder.providedBlockReportsCount).getValue();
        LOG.debug("HopsFS-Cloud. BR get counter : " + counter);

        return counter;
      }
    };
    return (Long) handler.handle();
  }

  public long getProvidedBlocksScanStartTime()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.GET_PROVIDED_BLOCK_CHECK_START_TIME) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        long time = (long) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime).getValue();

        if(LOG.isDebugEnabled()) {
          LOG.trace("HopsFS-Cloud. BR get start time: " +
                  new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(time)));
        }

        return time;
      }
    };
    return (Long) handler.handle();
  }

  private static void setProvidedBlocksScanStartTime(final long newTime)
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.UPDATE_PROVIDED_BLOCK_CHECK_START_TIME) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        Variables.updateVariable(
                new LongVariable(
                        Variable.Finder.providedBlocksCheckStartTime,
                        newTime
                ));
        LOG.debug("HopsFS-Cloud. BR set start time to: " + newTime);
        return null;
      }
    };
    handler.handle();
  }

  public static void deleteAllTask()
          throws IOException {
    HopsTransactionalRequestHandler handler = new HopsTransactionalRequestHandler(
            HDFSOperationType.BR_DELETE_ALL_TASKS) {
      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        //take a lock on HdfsVariables.providedBlocksCheckStartTime to sync
        HdfsStorageFactory.getConnector().writeLock();
        LongVariable var = (LongVariable) Variables.getVariable(
                Variable.Finder.providedBlocksCheckStartTime);
      }

      @Override
      public Object performTask() throws IOException {
        EntityManager.preventStorageCall(false);
        ProvidedBlockReportTasksDataAccess da = (ProvidedBlockReportTasksDataAccess) HdfsStorageFactory
                .getDataAccess(ProvidedBlockReportTasksDataAccess.class);
        da.deleteAll();
        LOG.debug("HopsFS-Cloud. BR Deleted all tasks");
        return null;
      }
    };
    handler.handle();
  }

  void handleGCSFailedMultipartUploads(String prefix) throws IOException{
    if (isGCSMultipartUpload){
      List<ActiveMultipartUploads> partialBlocks =
              cloudConnector.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                      prefix);
      for ( ActiveMultipartUploads pb : partialBlocks) {

        GCSActiveMultipartUploads gcsPB =  (GCSActiveMultipartUploads)pb;
        BlockInfoContiguous blockInfo = findBlockInfo(gcsPB.getId().getBlockID());
        if (blockInfo == null) {
          for (Blob blob: gcsPB.getParts() ){
            LOG.info("HopsFS-Cloud. BR Corresponding blockinfo not found for parts. " +
                    "Block " + gcsPB.getId().toString()+" Deleting Part: "+blob.getName());
            cloudConnector.deleteObject(blob.getBucket(), blob.getName());
          }
          continue;
        }

        if(blockInfo instanceof  BlockInfoContiguousUnderConstruction) {
          LOG.info("HopsFS-Cloud. Not deleting multipart as the Block is under construction." +
                  " Block " + gcsPB.getId().toString());
          continue;
        }

        // the  block is closed.
        if(blockInfo.isComplete()){
          for (Blob blob : gcsPB.getParts()) {
            LOG.info("HopsFS-Cloud. Block is complete. Deleting multipart. Part" + blob.getName()+
                    " Block " + gcsPB.getId().toString());
            cloudConnector.deleteObject(blob.getBucket(), blob.getName());
          }
        }
      }
    }
  }

  public BlockInfoContiguous findBlockInfo(final long bid)
          throws IOException {
    HopsTransactionalRequestHandler handler =
            new HopsTransactionalRequestHandler(HDFSOperationType.GET_BLOCK_BY_BID) {
              @Override
              public void acquireLock(TransactionLocks locks) throws IOException {
              }

              @Override
              public Object performTask() throws IOException {

                //get the primary keys of the blocks in the range
                BlockLookUpDataAccess da = (BlockLookUpDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockLookUpDataAccess.class);

                long allBlockIDs[] = new long[1];
                allBlockIDs[0] = bid;
                long allInodeIDs[] = da.findINodeIdsByBlockIds(allBlockIDs);
                if (allInodeIDs.length != 1) {
                  return  null;
                }

                BlockInfoDataAccess bda = (BlockInfoDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockInfoDataAccess.class);
                return bda.findById(bid, allInodeIDs[0]);
              }
            };
    return (BlockInfoContiguous) handler.handle();
  }

  void checkS3AbandonedBlocks() throws IOException {
    if(isS3MultipartUpload) {
      for (ActiveMultipartUploads u : cloudConnector.
              listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                      CloudHelper.ROOT_PREFIX)) {
        S3ActiveMultipartUploads upload = (S3ActiveMultipartUploads) u;
        long elapsedTime = System.currentTimeMillis() - upload.getStartTime();
        if (elapsedTime > deleteAbandonedBlocksAfter) {
          cloudConnector.abortMultipartUpload(upload.getBucket(), upload.getObjectID(),
                  upload.getUploadID());
        }
      }
    }
  }

  public boolean isBRInProgress() {
    return isBRInProgress;
  }
}
