/*
 * Copyright (C) 2021 Logical Clocks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.UnderReplicatedBlockDataAccess;
import io.hops.metadata.hdfs.entity.BlockInfoProjected;
import io.hops.metadata.hdfs.entity.UnderReplicatedBlock;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * Compares the file system metadata with the blocks stored in cloud
 * S3/Azure blob store
 * <p>
 * If a block exists in the metadata and in the cloud then no worries
 * If a block exists in metadata but not in the cloud
 * then try to recover the block using an old version.
 * panic if the block is not recoverable.
 * Delete all blocks from the cloud that are no longer part of namespace
 */

public class HopsFSRestore {
  public static final Log LOG = LogFactory.getLog(HopsFSRestore.class);
  private CloudPersistenceProvider cloudConnector;
  private ExecutorService executorService;
  private Map<BlockIDAndGSTuple, BlockInfoProjected> dbBlocks;
  private Map<BlockIDAndGSTuple, CloudBlock> cloudBlocks;
  private final List<String> buckets;
  private final int prefixSize;

  public HopsFSRestore(Configuration conf) throws IOException {
    this.cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    this.executorService = Executors.newFixedThreadPool(
            conf.getInt(DFSConfigKeys.DFS_CLOUD_BACKUP_RESTORE_THREADS_KEY,
                    DFSConfigKeys.DFS_CLOUD_BACKUP_RESTORE_THREADS_DEFAUlT));
    this.buckets = CloudHelper.getBucketsFromConf(conf);

    if (buckets.size() != 1) {
      throw new IllegalStateException("There should be exactly one bucket/container" +
              " configured in hdfs-site.xml");
    }

    this.prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
  }

  public RestoreStats compareAndFixState() throws IOException, ExecutionException, InterruptedException {

    if (!cloudConnector.isVersioningSupported(buckets.get(0))) {
      throw new UnsupportedOperationException("HopsFS can not restore data as the " +
              "bucket/container does not support Versioning.");
    }

    RestoreStats stats = new RestoreStats();

    cloudBlocks = getExistingBlocksFromCloud();
    dbBlocks = getExistingBlocksFromDB();

    //---------------------------------------------------
    // Step 1. Remove the blocks that match. This is fast
    long startTime = System.currentTimeMillis();
    long matching = 0;
    Iterator<BlockIDAndGSTuple> iter = dbBlocks.keySet().iterator();
    while (iter.hasNext()) {
      BlockIDAndGSTuple blkID = iter.next();

      if (cloudBlocks.containsKey(blkID)) {
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Block ID: " + blkID + " exists in both DB and Cloud ");
        }

        iter.remove();
        cloudBlocks.remove(blkID);
        matching++;
      }
      // else the block is deleted or missing altogether
    }
    stats.setMatching(matching);
    LOG.info("HopsFS-Cloud. " + matching + " blocks exists in both the DB and Cloud. Time taken: " +
            (System.currentTimeMillis() - startTime) + " ms.");

    //---------------------------------------------------
    // Step 2. Try to recover the block using cloud versioning
    // Using multiple threads as this can be slow
    if (dbBlocks.size() > 0) {
      LOG.info("HopsFS-Cloud. " + dbBlocks.size() + " blocks left unaccounted for. Recovering blocks ...");

      startTime = System.currentTimeMillis();
      List<Future> futures = new ArrayList<>(dbBlocks.size());
      for (BlockIDAndGSTuple id : dbBlocks.keySet()) {
        futures.add(executorService.submit(new BlockRecoverer(id)));
      }

      long recovered = 0;
      for (Future future : futures) {
        BlockIDAndGSTuple blkID = (BlockIDAndGSTuple) future.get();
        if (blkID.getBlockID() != Block.NON_EXISTING_BLK_ID) {
          dbBlocks.remove(blkID);
          cloudBlocks.remove(blkID);
          recovered++;
        } // else the block is missing from cloud
      }

      stats.setRecovered(recovered);
      LOG.info("HopsFS-Cloud. " + recovered + " blocks successfully recovered. Time taken: " +
              (System.currentTimeMillis() - startTime) + " ms.");

      // *MISSING BLOCKS*
      stats.setMissingFromCloud(dbBlocks.size());
      if (dbBlocks.size() > 0) {
        LOG.warn("HopsFS-Cloud. " + dbBlocks.size() + " blocks are missing from cloud.");
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Missing blocks IDs : " + Arrays.toString(dbBlocks.keySet().toArray()));
        }
        handleMissingBlocks(dbBlocks);
      }
    }

    //---------------------------------------------------
    // Step 3. Now delete the extra blocks in cloud
    // that are not needed any more
    // Using multiple threads as this can be slow
    if (cloudBlocks.size() > 0) {
      List<Future> futures = new ArrayList<Future>(cloudBlocks.size());
      startTime = System.currentTimeMillis();
      long deleted = 0;
      for (BlockIDAndGSTuple blockID : cloudBlocks.keySet()) {
          CloudBlock block = cloudBlocks.get(blockID);
          futures.add(executorService.submit(new BlockDeleter(block.getBlock())));
      }

      for (Future future : futures) {
        future.get();
        deleted++;
      }
      stats.setExcessCloudBlocks(deleted);
      LOG.info("HopsFS-Cloud. " + deleted + " blocks successfully deleted from Cloud. Time taken:" +
              " " +
              (System.currentTimeMillis() - startTime) + " ms.");
    }

    return stats;
  }

  private void handleMissingBlocks(Map<BlockIDAndGSTuple, BlockInfoProjected> dbBlocks) throws ExecutionException, InterruptedException {
    long startTime = System.currentTimeMillis();
    List<Future> futures = new ArrayList<Future>(cloudBlocks.size());
    for(BlockInfoProjected block : dbBlocks.values()){
      futures.add(executorService.submit(new MissingBlockHandler(block)));
    }

    for (Future future : futures) {
      future.get();
    }
    LOG.info("HopsFS-Cloud. " + dbBlocks.size() + " blocks successfully added to missing list. " +
            "Time taken: " + (System.currentTimeMillis() - startTime) + " ms.");
  }

  private Map<BlockIDAndGSTuple, BlockInfoProjected> getExistingBlocksFromDB() throws IOException {
    long startTime = System.currentTimeMillis();
    Map<BlockIDAndGSTuple, BlockInfoProjected> blocksMap =
            (Map<BlockIDAndGSTuple, BlockInfoProjected>) new LightWeightRequestHandler(
                    HDFSOperationType.GET_ALL_PROVIDED_BLOCKS) {
              @Override
              public Object performTask() throws IOException {
                BlockInfoDataAccess da = (BlockInfoDataAccess) HdfsStorageFactory.getDataAccess(BlockInfoDataAccess.class);
                return da.getAllProvidedBlocksIDs();
              }
            }.handle();

    LOG.info("HopsFS-Cloud: Reading all the blocks ( Size: " + blocksMap.size() +
            " ) from DB took " + (System.currentTimeMillis() - startTime) + " ms.");
    return blocksMap;
  }

  @VisibleForTesting
  public Map<BlockIDAndGSTuple, CloudBlock> getExistingBlocksFromCloud() throws IOException {
    long startTime = System.currentTimeMillis();
    List<String> dirs = cloudConnector.getAllHopsFSDirectories(buckets);
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Total Prefixes : " + dirs);
    }

    Map<BlockIDAndGSTuple, CloudBlock> cloudBlocks = new HashMap<>();

    List<Future> futures = new ArrayList<>();
    for (String dir : dirs) {
      futures.add(executorService.submit(new ListCloudPrefixs(dir)));
    }

    for (Future f : futures) {
      try {
        Map<BlockIDAndGSTuple, CloudBlock> blocksSubSet = (Map<BlockIDAndGSTuple, CloudBlock>) f.get();
        cloudBlocks.putAll(blocksSubSet);

      } catch (ExecutionException e) {
        LOG.error("Exception was thrown during listing cloud storage", e);
        Throwable throwable = e.getCause();
        if (throwable instanceof IOException) {
          throw (IOException) throwable;
        } else {
          throw new IOException(e);
        }
      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    }

    LOG.info("HopsFS-Cloud: Reading all the blocks ( Size: " + cloudBlocks.size() +
            " ) from cloud took " + (System.currentTimeMillis() - startTime) + " ms.");
    return cloudBlocks;
  }

  private boolean recoverDeletedBlock(long blockID, long gs) throws IOException {
    Block blockTmp = new Block(blockID);
    blockTmp.setGenerationStampNoPersistance(gs);

    return cloudConnector.restoreDeletedBlock(buckets.get(0),
            CloudHelper.getBlockKey(prefixSize, blockTmp))
            &&
            cloudConnector.restoreDeletedBlock(buckets.get(0),
                    CloudHelper.getMetaFileKey(prefixSize, blockTmp));
  }

  private void deleteCloudBlock(Block block) throws IOException {
    cloudConnector.deleteObject(buckets.get(0), CloudHelper.getBlockKey(prefixSize, block));
    cloudConnector.deleteObject(buckets.get(0), CloudHelper.getMetaFileKey(prefixSize, block));
  }


  private class ListCloudPrefixs implements Callable<Map<BlockIDAndGSTuple, CloudBlock>> {
    private final String prefix;

    ListCloudPrefixs(String prefix) {
      this.prefix = prefix;
    }

    @Override
    public Map<BlockIDAndGSTuple, CloudBlock> call() throws Exception {
      return cloudConnector.getAll(prefix, buckets);
    }
  }

  private class BlockRecoverer implements Callable<BlockIDAndGSTuple> {
    private final BlockIDAndGSTuple id;

    BlockRecoverer(BlockIDAndGSTuple id) {
      this.id = id;
    }

    @Override
    public BlockIDAndGSTuple call() throws Exception {
      if (recoverDeletedBlock(id.getBlockID(), id.getGs())) {
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Block ID: " + id + " recovered from an old version");
        }
        return id;
      } else {
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Block ID: " + id + " Failed to recover from an old " +
                  "version");
        }
        return new BlockIDAndGSTuple(Block.NON_EXISTING_BLK_ID, Block.NON_EXISTING_BLK_GS);
      }
    }
  }

  private class BlockDeleter implements Callable {
    private final Block block;

    BlockDeleter(Block block) {
      this.block = block;
    }

    @Override
    public Object call() throws Exception {
      deleteCloudBlock(block);
      return null;
    }
  }


  private class MissingBlockHandler implements Callable {
    private final BlockInfoProjected block;

    MissingBlockHandler(BlockInfoProjected block) {
      this.block = block;
    }

    @Override
    public Object call() throws Exception {
      new LightWeightRequestHandler(
              HDFSOperationType.ADD_UNDER_REPLICATED_BLOCK) {
        @Override
        public Object performTask() throws IOException {
          UnderReplicatedBlockDataAccess da =
                  (UnderReplicatedBlockDataAccess) HdfsStorageFactory.getDataAccess(UnderReplicatedBlockDataAccess.class);
          UnderReplicatedBlock urb =
                  new UnderReplicatedBlock(UnderReplicatedBlocks.QUEUE_WITH_CORRUPT_BLOCKS, block.getBlockId(),
                          block.getInodeId(), 1);
          List<UnderReplicatedBlock> newBlks = new ArrayList<>();
          newBlks.add(urb);
          da.prepare(Collections.EMPTY_LIST, newBlks, Collections.EMPTY_LIST);
          return null;
        }
      }.handle();
      return null;
    }
  }

  public class RestoreStats {
    private long matching;
    private long recovered;
    private long missingFromCloud;
    private long excessCloudBlocks;

    public long getMatching() {
      return matching;
    }

    public void setMatching(long matching) {
      this.matching = matching;
    }

    public long getRecovered() {
      return recovered;
    }

    public void setRecovered(long recovered) {
      this.recovered = recovered;
    }

    public long getMissingFromCloud() {
      return missingFromCloud;
    }

    public void setMissingFromCloud(long missingFromCloud) {
      this.missingFromCloud = missingFromCloud;
    }

    public long getExcessCloudBlocks() {
      return excessCloudBlocks;
    }

    public void setExcessCloudBlocks(long excessCloudBlocks) {
      this.excessCloudBlocks = excessCloudBlocks;
    }

    @Override
    public String toString(){
      return "Matching: "+matching+
              " Recoverd: "+recovered+
              " Missing from cloud:"+missingFromCloud+
              " Excess cloud blocks: "+excessCloudBlocks;

    }
  }
}
