/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.hops.transaction.context;

import io.hops.exception.StorageCallPreventedException;
import io.hops.exception.StorageException;
import io.hops.exception.TransactionContextException;
import io.hops.metadata.common.FinderType;
import io.hops.metadata.hdfs.dal.ProvidedBlockCacheLocDataAccess;
import io.hops.metadata.hdfs.entity.ProvidedBlockCacheLoc;
import io.hops.transaction.lock.TransactionLocks;
import org.apache.commons.collections.map.HashedMap;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;

import java.util.Collection;
import java.util.Map;

public class ProvidedBlockCacheLocContext
        extends BaseEntityContext<Long, ProvidedBlockCacheLoc> {
  private ProvidedBlockCacheLocDataAccess dataAccess;
  private final Map<Long, ProvidedBlockCacheLoc> nullLocs = new HashedMap();

  public ProvidedBlockCacheLocContext(ProvidedBlockCacheLocDataAccess dataAccess) {
    this.dataAccess = dataAccess;
  }

  @Override
  public void update(ProvidedBlockCacheLoc loc)
          throws TransactionContextException {
    super.update(loc);

    if(nullLocs.containsKey(loc.getBlockID())){
      nullLocs.remove(loc.getBlockID());
    }

    if(isLogTraceEnabled()) {
      log("updated-provided-block-cache-loc", "bid",
              loc.getBlockID(), "sid", loc.getStorageID());
    }
  }

  @Override
  public void remove(ProvidedBlockCacheLoc loc)
          throws TransactionContextException {
    super.remove(loc);
    if(isLogTraceEnabled()) {
      log("removed-provided-block-cache-loc", "bid", loc.getBlockID(),
              "sid", loc.getStorageID());
    }
  }

  @Override
  public ProvidedBlockCacheLoc find(FinderType<ProvidedBlockCacheLoc> finder, Object... params)
          throws TransactionContextException, StorageException {
    ProvidedBlockCacheLoc.Finder clfinder = (ProvidedBlockCacheLoc.Finder) finder;
    switch (clfinder) {
      case ByBlockId:
        return findByPrimaryKey(clfinder, params);
      default:
        throw new RuntimeException(UNSUPPORTED_FINDER);
    }
  }

  @Override
  public Collection<ProvidedBlockCacheLoc> findList(FinderType<ProvidedBlockCacheLoc> finder, Object... params)
          throws TransactionContextException, StorageException {
    ProvidedBlockCacheLoc.Finder clFinder = (ProvidedBlockCacheLoc.Finder) finder;
    switch (clFinder) {
      case ByBlockIds:
        return batchFinder(clFinder, params);
      default:
        throw new RuntimeException(UNSUPPORTED_FINDER);
    }
  }

  private ProvidedBlockCacheLoc findByPrimaryKey(ProvidedBlockCacheLoc.Finder hbFinder, Object[] params)
          throws StorageException, StorageCallPreventedException {
    Long blockId = (Long) params[0];
    ProvidedBlockCacheLoc result;
    if (contains(blockId)) {
      result = get(blockId);
      hit(hbFinder, result, "bid", blockId);
    } else if (nullLocs.containsKey(blockId)) {
      return null;
    } else {
      aboutToAccessStorage(hbFinder, params);
      result = (ProvidedBlockCacheLoc) dataAccess.findByBlockID(blockId);
      if(result != null){
        gotFromDB(blockId, result);
      } else {
        nullLocs.put(blockId, null);
      }
      miss(hbFinder, result, "bid", blockId);
    }
    return result;
  }

  private Collection<ProvidedBlockCacheLoc> batchFinder(ProvidedBlockCacheLoc.Finder hbFinder,
                                                        Object[] params)
          throws StorageException, StorageCallPreventedException {
    Long[] blockIds = (Long[]) params;
    Map<Long, ProvidedBlockCacheLoc> result;
    aboutToAccessStorage(hbFinder, params);
    result = dataAccess.findByBlockIDs(blockIds);
    for(long blkId : result.keySet()){
      ProvidedBlockCacheLoc loc  = result.get(blkId);
      gotFromDB(blkId, loc);
      miss(hbFinder, loc, "bid", blkId);
    }
    for(long blkId : blockIds){
      if(!result.containsKey(blkId)){
        nullLocs.put(blkId, null);
      }
    }
    return result.values();
  }

  @Override
  public void prepare(TransactionLocks tlm)
          throws TransactionContextException, StorageException {
    dataAccess.prepare(getRemoved(), getAdded(), getModified());
  }

  @Override
  Long getKey(ProvidedBlockCacheLoc loc) {
    return loc.getBlockID();
  }
}